import React, { useState, useEffect, useMemo } from 'react';
import { Sidebar } from './components/Sidebar';
import { StatCard } from './components/StatCard';
import { Product, Invoice, CartItem, Customer, Category, Supplier, User, StoreSettings } from './types';
import { 
  Package, 
  AlertTriangle, 
  DollarSign, 
  TrendingUp, 
  Trash2, 
  Edit, 
  Plus, 
  Search, 
  ShoppingCart, 
  Printer, 
  Save,
  X,
  Users,
  Tags,
  Truck,
  UserCog,
  Calendar,
  BarChart3,
  Lock,
  LogIn,
  Settings as SettingsIcon,
  Store,
  Phone,
  MapPin,
  Coins,
  ScanBarcode,
  ChevronDown
} from 'lucide-react';

// --- Initial Mock Data ---
const initialProducts: Product[] = [
  { id: '1', name: 'لابتوب ديل XPS', barcode: '123456789', category: 'الكترونيات', price: 4500, cost: 3800, stock: 5, minStock: 2 },
  { id: '2', name: 'ماوس لاسلكي لوجيتك', barcode: '987654321', category: 'ملحقات', price: 120, cost: 80, stock: 15, minStock: 5 },
  { id: '3', name: 'شاشة سامسونج 27 بوصة', category: 'الكترونيات', price: 850, cost: 600, stock: 3, minStock: 3 },
  { id: '4', name: 'كابل HDMI 2.0', category: 'كابلات', price: 45, cost: 15, stock: 50, minStock: 10 },
  { id: '5', name: 'طابعة اتش بي ليزر', category: 'طابعات', price: 1200, cost: 950, stock: 1, minStock: 2 },
];

const initialCustomers: Customer[] = [
  { id: '1', name: 'عميل عام', phone: '', email: '' },
  { id: '2', name: 'شركة التقنية الحديثة', phone: '0501234567', email: 'info@tech.com' },
];

const initialCategories: Category[] = [
  { id: '1', name: 'الكترونيات', description: 'أجهزة ومعدات الكترونية' },
  { id: '2', name: 'ملحقات', description: 'اكسسوارات وكماليات' },
  { id: '3', name: 'طابعات', description: 'طابعات ومستلزمات طباعة' },
];

const initialSuppliers: Supplier[] = [
  { id: '1', name: 'مؤسسة التوريد السريع', contactPerson: 'سعيد', phone: '0555555555' },
];

const initialUsers: User[] = [
  { id: '1', fullName: 'المدير العام', username: 'admin', password: '123', role: 'admin' },
  { id: '2', fullName: 'موظف الكاشير', username: 'cashier', password: '123', role: 'staff' },
];

const App: React.FC = () => {
  // --- Auth State ---
  const [isAuthenticated, setIsAuthenticated] = useState(false);
  const [currentUser, setCurrentUser] = useState<User | null>(null);
  const [loginUsername, setLoginUsername] = useState('');
  const [loginPassword, setLoginPassword] = useState('');
  const [loginError, setLoginError] = useState('');

  const [currentTab, setCurrentTab] = useState('dashboard');
  
  // --- Data State ---
  const [products, setProducts] = useState<Product[]>(initialProducts);
  const [invoices, setInvoices] = useState<Invoice[]>([]);
  const [customers, setCustomers] = useState<Customer[]>(initialCustomers);
  const [categories, setCategories] = useState<Category[]>(initialCategories);
  const [suppliers, setSuppliers] = useState<Supplier[]>(initialSuppliers);
  const [users, setUsers] = useState<User[]>(initialUsers);
  
  // --- Settings State ---
  const [storeSettings, setStoreSettings] = useState<StoreSettings>({
    name: 'مخزوني برو',
    address: 'الدار البيضاء، المغرب',
    currency: 'MAD',
    phone1: '0522000000',
    phone2: ''
  });

  // --- Report State ---
  const [reportStartDate, setReportStartDate] = useState(new Date().toISOString().split('T')[0]);
  const [reportEndDate, setReportEndDate] = useState(new Date().toISOString().split('T')[0]);

  // --- Temporary Input States for New Entities ---
  const [newCustomer, setNewCustomer] = useState<Partial<Customer>>({});
  const [newCategory, setNewCategory] = useState<Partial<Category>>({});
  const [newSupplier, setNewSupplier] = useState<Partial<Supplier>>({});
  const [newUser, setNewUser] = useState<Partial<User>>({ role: 'staff' });

  // --- Inventory State ---
  const [showProductModal, setShowProductModal] = useState(false);
  const [editingProduct, setEditingProduct] = useState<Product | null>(null);
  const [tempProduct, setTempProduct] = useState<Partial<Product>>({});
  const [searchTerm, setSearchTerm] = useState('');
  
  // --- Delete Modal State ---
  const [showDeleteModal, setShowDeleteModal] = useState(false);
  const [productToDelete, setProductToDelete] = useState<string | null>(null);

  // --- Invoice State ---
  const [cart, setCart] = useState<CartItem[]>([]);
  const [selectedCustomer, setSelectedCustomer] = useState<string>('عميل عام');
  const [invoiceDiscount, setInvoiceDiscount] = useState(0);
  // --- Customer Search State ---
  const [isCustomerDropdownOpen, setIsCustomerDropdownOpen] = useState(false);
  const [customerSearchQuery, setCustomerSearchQuery] = useState('');

  // --- Derived Stats ---
  const stats = useMemo(() => {
    const totalValue = products.reduce((acc, curr) => acc + (curr.price * curr.stock), 0);
    const lowStock = products.filter(p => p.stock <= p.minStock).length;
    const totalSales = invoices.reduce((acc, curr) => acc + curr.finalAmount, 0);
    return {
      totalProducts: products.length,
      lowStockCount: lowStock,
      totalInventoryValue: totalValue,
      totalSales: totalSales
    };
  }, [products, invoices]);

  // --- Report Logic ---
  const reportData = useMemo(() => {
    const start = new Date(reportStartDate);
    start.setHours(0, 0, 0, 0);
    const end = new Date(reportEndDate);
    end.setHours(23, 59, 59, 999);

    const filteredInvoices = invoices.filter(inv => {
      const invDate = inv.timestamp ? new Date(inv.timestamp) : new Date(); 
      return invDate >= start && invDate <= end;
    });

    const totalSales = filteredInvoices.reduce((sum, inv) => sum + inv.finalAmount, 0);
    const totalInvoices = filteredInvoices.length;
    const averageOrderValue = totalInvoices > 0 ? totalSales / totalInvoices : 0;

    const dailyData: Record<string, number> = {};
    filteredInvoices.forEach(inv => {
        const dateKey = inv.timestamp 
            ? new Date(inv.timestamp).toISOString().split('T')[0] 
            : inv.date;
        dailyData[dateKey] = (dailyData[dateKey] || 0) + inv.finalAmount;
    });

    const chartData = Object.keys(dailyData).sort().map(date => ({
        date,
        amount: dailyData[date]
    }));

    return { totalSales, totalInvoices, averageOrderValue, chartData };
  }, [invoices, reportStartDate, reportEndDate]);

  // --- Helpers ---
  const formatCurrency = (val: number) => {
    try {
      return new Intl.NumberFormat('ar-MA', { style: 'currency', currency: storeSettings.currency }).format(val);
    } catch (e) {
      // Fallback if currency code is invalid
      return `${val} ${storeSettings.currency}`;
    }
  };

  const getSupplierName = (supplierId?: string) => {
    if (!supplierId) return '-';
    const supplier = suppliers.find(s => s.id === supplierId);
    return supplier ? supplier.name : '-';
  };

  // --- Auth Handlers ---
  const handleLogin = (e: React.FormEvent) => {
    e.preventDefault();
    const user = users.find(u => u.username === loginUsername && u.password === loginPassword);
    if (user) {
      setCurrentUser(user);
      setIsAuthenticated(true);
      setLoginError('');
      setLoginUsername('');
      setLoginPassword('');
    } else {
      setLoginError('اسم المستخدم أو كلمة المرور غير صحيحة');
    }
  };

  const handleLogout = () => {
    setIsAuthenticated(false);
    setCurrentUser(null);
    setCurrentTab('dashboard');
  };

  // --- Handlers: Print ---
  const handlePrintInvoice = (invoice: Invoice) => {
    const printWindow = window.open('', '_blank');
    if (!printWindow) return;

    const invoiceHtml = `
      <!DOCTYPE html>
      <html dir="rtl" lang="ar">
        <head>
          <meta charset="UTF-8">
          <title>فاتورة #${invoice.id.slice(-6)}</title>
          <link href="https://fonts.googleapis.com/css2?family=Cairo:wght@400;600;700&display=swap" rel="stylesheet">
          <style>
            body { font-family: 'Cairo', sans-serif; padding: 20px; direction: rtl; }
            .header { text-align: center; margin-bottom: 20px; border-bottom: 2px solid #eee; padding-bottom: 20px; }
            .title { font-size: 24px; font-weight: bold; color: #333; margin-bottom: 5px; }
            .subtitle { color: #666; font-size: 14px; margin-bottom: 3px; }
            .contact { color: #555; font-size: 14px; font-weight: 600; }
            .info-grid { display: grid; grid-template-columns: 1fr 1fr; gap: 20px; margin-bottom: 30px; margin-top: 20px; }
            .info-item { margin-bottom: 5px; }
            .label { font-weight: bold; color: #555; }
            table { width: 100%; border-collapse: collapse; margin-bottom: 30px; }
            th { background-color: #f8f9fa; padding: 12px; text-align: right; border-bottom: 2px solid #ddd; font-weight: bold; }
            td { padding: 12px; border-bottom: 1px solid #eee; }
            .totals { margin-top: 20px; border-top: 2px solid #eee; padding-top: 20px; }
            .total-row { display: flex; justify-content: space-between; margin-bottom: 10px; }
            .final-total { font-size: 1.2em; font-weight: bold; color: #10b981; }
            .footer { text-align: center; margin-top: 50px; color: #888; font-size: 0.9em; }
            @media print {
              @page { margin: 0; }
              body { padding: 20px; }
              .no-print { display: none; }
            }
          </style>
        </head>
        <body>
          <div class="header">
            <div class="title">${storeSettings.name}</div>
            <div class="subtitle">${storeSettings.address}</div>
            <div class="contact">
               هاتف: ${storeSettings.phone1} 
               ${storeSettings.phone2 ? ' | ' + storeSettings.phone2 : ''}
            </div>
          </div>
          
          <div class="info-grid">
            <div>
              <div class="info-item"><span class="label">رقم الفاتورة:</span> #${invoice.id.slice(-6)}</div>
              <div class="info-item"><span class="label">التاريخ:</span> ${invoice.date}</div>
            </div>
            <div>
              <div class="info-item"><span class="label">العميل:</span> ${invoice.customerName}</div>
              <div class="info-item"><span class="label">البائع:</span> ${currentUser?.fullName || 'غير محدد'}</div>
            </div>
          </div>

          <table>
            <thead>
              <tr>
                <th>المنتج</th>
                <th>الكمية</th>
                <th>السعر</th>
                <th>الإجمالي</th>
              </tr>
            </thead>
            <tbody>
              ${invoice.items.map(item => `
                <tr>
                  <td>${item.name}</td>
                  <td>${item.quantity}</td>
                  <td>${formatCurrency(item.price)}</td>
                  <td>${formatCurrency(item.price * item.quantity)}</td>
                </tr>
              `).join('')}
            </tbody>
          </table>

          <div class="totals">
            <div class="total-row">
              <span>المجموع الفرعي:</span>
              <span>${formatCurrency(invoice.totalAmount)}</span>
            </div>
            <div class="total-row" style="color: #ef4444;">
              <span>الخصم:</span>
              <span>${formatCurrency(invoice.discount)}</span>
            </div>
            <div class="total-row final-total">
              <span>الإجمالي النهائي:</span>
              <span>${formatCurrency(invoice.finalAmount)}</span>
            </div>
          </div>

          <div class="footer">
            <p>شكرًا لتسوقكم معنا!</p>
            <p>${new Date().toLocaleString('ar-MA')}</p>
          </div>
          <script>
            window.onload = () => { window.print(); setTimeout(() => window.close(), 1000); };
          </script>
        </body>
      </html>
    `;

    printWindow.document.write(invoiceHtml);
    printWindow.document.close();
  };

  // --- Handlers: Inventory ---
  const handleSaveProduct = () => {
    if (!tempProduct.name || !tempProduct.price) return;

    if (editingProduct) {
      setProducts(products.map(p => p.id === editingProduct.id ? { ...editingProduct, ...tempProduct } as Product : p));
    } else {
      const newProduct: Product = {
        id: Date.now().toString(),
        name: tempProduct.name || '',
        barcode: tempProduct.barcode,
        category: tempProduct.category || 'عام',
        price: Number(tempProduct.price) || 0,
        cost: Number(tempProduct.cost) || 0,
        stock: Number(tempProduct.stock) || 0,
        minStock: Number(tempProduct.minStock) || 0,
        supplierId: tempProduct.supplierId,
      };
      setProducts([...products, newProduct]);
    }
    setShowProductModal(false);
    setEditingProduct(null);
    setTempProduct({});
  };

  const handleDeleteProduct = (id: string) => {
    setProductToDelete(id);
    setShowDeleteModal(true);
  };

  const confirmDeleteProduct = () => {
    if (productToDelete) {
      setProducts(products.filter(p => p.id !== productToDelete));
      setProductToDelete(null);
      setShowDeleteModal(false);
    }
  };

  // --- Handlers: New Entities CRUD ---
  const handleAddCustomer = () => {
    if (!newCustomer.name) return;
    setCustomers([...customers, { id: Date.now().toString(), name: newCustomer.name, phone: newCustomer.phone || '', email: newCustomer.email || '' }]);
    setNewCustomer({});
  };
  const deleteCustomer = (id: string) => setCustomers(customers.filter(c => c.id !== id));

  const handleAddCategory = () => {
    if (!newCategory.name) return;
    setCategories([...categories, { id: Date.now().toString(), name: newCategory.name, description: newCategory.description || '' }]);
    setNewCategory({});
  };
  const deleteCategory = (id: string) => setCategories(categories.filter(c => c.id !== id));

  const handleAddSupplier = () => {
    if (!newSupplier.name) return;
    setSuppliers([...suppliers, { id: Date.now().toString(), name: newSupplier.name, contactPerson: newSupplier.contactPerson || '', phone: newSupplier.phone || '' }]);
    setNewSupplier({});
  };
  const deleteSupplier = (id: string) => setSuppliers(suppliers.filter(c => c.id !== id));

  const handleAddUser = () => {
    if (!newUser.username || !newUser.password || !newUser.fullName) return;
    setUsers([...users, { 
        id: Date.now().toString(), 
        fullName: newUser.fullName,
        username: newUser.username, 
        password: newUser.password,
        role: newUser.role as 'admin' | 'staff' 
    }]);
    setNewUser({ role: 'staff' });
  };
  const deleteUser = (id: string) => setUsers(users.filter(c => c.id !== id));

  // --- Handlers: Invoice ---
  const addToCart = (product: Product) => {
    const existing = cart.find(item => item.id === product.id);
    if (existing) {
      if (existing.quantity < product.stock) {
        setCart(cart.map(item => item.id === product.id ? { ...item, quantity: item.quantity + 1 } : item));
      } else {
        alert('الكمية المطلوبة غير متوفرة في المخزون');
      }
    } else {
      if (product.stock > 0) {
        setCart([...cart, { ...product, quantity: 1 }]);
      } else {
        alert('المنتج نفد من المخزون');
      }
    }
  };

  const removeFromCart = (id: string) => {
    setCart(cart.filter(item => item.id !== id));
  };

  const updateCartQuantity = (id: string, delta: number) => {
    setCart(cart.map(item => {
      if (item.id === id) {
        const product = products.find(p => p.id === id);
        const newQty = Math.max(1, item.quantity + delta);
        if (product && newQty <= product.stock) {
          return { ...item, quantity: newQty };
        }
      }
      return item;
    }));
  };

  const handleSaveInvoice = () => {
    if (cart.length === 0) return;

    const totalAmount = cart.reduce((sum, item) => sum + (item.price * item.quantity), 0);
    const finalAmount = Math.max(0, totalAmount - invoiceDiscount);

    const now = new Date();
    const newInvoice: Invoice = {
      id: Date.now().toString(),
      customerName: selectedCustomer || 'عميل نقدي',
      date: now.toLocaleDateString('ar-MA'),
      timestamp: now.getTime(),
      items: [...cart],
      totalAmount,
      discount: invoiceDiscount,
      finalAmount
    };

    const newProducts = products.map(p => {
      const cartItem = cart.find(c => c.id === p.id);
      if (cartItem) {
        return { ...p, stock: p.stock - cartItem.quantity };
      }
      return p;
    });

    setProducts(newProducts);
    setInvoices([newInvoice, ...invoices]);
    setCart([]);
    setSelectedCustomer('عميل عام');
    setInvoiceDiscount(0);
    setCurrentTab('invoices');
    
    // Automatically print the invoice after saving
    handlePrintInvoice(newInvoice);
  };


  const renderChart = () => {
    if (reportData.chartData.length === 0) {
        return <div className="h-64 flex items-center justify-center text-slate-400">لا توجد بيانات للعرض في هذه الفترة</div>;
    }
    const maxAmount = Math.max(...reportData.chartData.map(d => d.amount));
    return (
        <div className="h-64 flex items-end gap-2 mt-4 px-2">
            {reportData.chartData.map((item, index) => {
                const heightPercentage = maxAmount > 0 ? (item.amount / maxAmount) * 100 : 0;
                return (
                    <div key={index} className="flex-1 flex flex-col items-center group relative">
                        <div className="absolute bottom-full mb-2 hidden group-hover:block bg-slate-800 text-white text-xs p-2 rounded z-10 whitespace-nowrap">
                            {item.date}: {formatCurrency(item.amount)}
                        </div>
                        <div style={{ height: `${Math.max(heightPercentage, 5)}%` }} className="w-full bg-indigo-500 rounded-t-md opacity-80 hover:opacity-100 transition-all cursor-pointer"></div>
                        <span className="text-[10px] text-slate-500 mt-1 rotate-45 origin-left truncate w-full text-center">{item.date.slice(5)}</span>
                    </div>
                )
            })}
        </div>
    );
  };

  // --- Login Screen ---
  if (!isAuthenticated) {
    return (
      <div className="min-h-screen bg-slate-900 flex items-center justify-center p-4 font-[Cairo]" dir="rtl">
        <div className="bg-white rounded-2xl shadow-2xl w-full max-w-md overflow-hidden">
          <div className="p-8 bg-slate-50 border-b flex flex-col items-center">
             <div className="w-16 h-16 bg-emerald-100 rounded-full flex items-center justify-center mb-4">
                <Lock className="w-8 h-8 text-emerald-600" />
             </div>
             <h1 className="text-2xl font-bold text-slate-800">تسجيل الدخول</h1>
             <p className="text-slate-500 mt-1">مخزوني برو - نظام الإدارة المتكامل</p>
          </div>
          
          <form onSubmit={handleLogin} className="p-8 space-y-6">
            {loginError && (
              <div className="bg-red-50 text-red-600 p-3 rounded-lg text-sm flex items-center gap-2">
                <AlertTriangle size={16} />
                {loginError}
              </div>
            )}
            
            <div>
              <label className="block text-sm font-medium text-slate-700 mb-1">اسم المستخدم</label>
              <input 
                type="text" 
                className="w-full px-4 py-3 border rounded-lg focus:ring-2 focus:ring-emerald-500 outline-none transition-all"
                placeholder="ادخل اسم المستخدم..."
                value={loginUsername}
                onChange={(e) => setLoginUsername(e.target.value)}
              />
            </div>
            
            <div>
              <label className="block text-sm font-medium text-slate-700 mb-1">كلمة المرور</label>
              <input 
                type="password" 
                className="w-full px-4 py-3 border rounded-lg focus:ring-2 focus:ring-emerald-500 outline-none transition-all"
                placeholder="ادخل كلمة المرور..."
                value={loginPassword}
                onChange={(e) => setLoginPassword(e.target.value)}
              />
            </div>
            
            <button 
              type="submit"
              className="w-full bg-emerald-600 hover:bg-emerald-700 text-white font-bold py-3 rounded-lg transition-colors flex items-center justify-center gap-2"
            >
              <LogIn size={20} />
              دخول للنظام
            </button>

            <div className="text-center text-xs text-slate-400 mt-4">
               بيانات تجريبية: admin / 123
            </div>
          </form>
        </div>
      </div>
    );
  }

  // --- Main App ---
  return (
    <div className="flex bg-slate-50 min-h-screen font-[Cairo]" dir="rtl">
      <Sidebar 
         currentTab={currentTab} 
         setCurrentTab={setCurrentTab} 
         onLogout={handleLogout}
         currentUser={currentUser}
      />

      <main className="flex-1 mr-64 p-8 transition-all">
        {/* --- DASHBOARD --- */}
        {currentTab === 'dashboard' && (
          <div className="space-y-6">
            <h2 className="text-3xl font-bold text-slate-800 mb-6">لوحة التحكم</h2>
            
            <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-6">
              <StatCard title="إجمالي المنتجات" value={stats.totalProducts} icon={Package} color="bg-blue-500" />
              <StatCard title="قيمة المخزون" value={formatCurrency(stats.totalInventoryValue)} icon={DollarSign} color="bg-emerald-500" />
              <StatCard title="تنبيهات المخزون" value={stats.lowStockCount} icon={AlertTriangle} color="bg-red-500" description="منتجات أوشكت على النفاد" />
              <StatCard title="إجمالي المبيعات" value={formatCurrency(stats.totalSales)} icon={TrendingUp} color="bg-purple-500" />
            </div>

            <div className="grid grid-cols-1 lg:grid-cols-2 gap-6 mt-8">
              <div className="bg-white p-6 rounded-xl shadow-sm border border-slate-100">
                <h3 className="text-lg font-bold mb-4 flex items-center gap-2">
                  <AlertTriangle className="text-amber-500 w-5 h-5" />
                  منتجات منخفضة المخزون
                </h3>
                {products.filter(p => p.stock <= p.minStock).length === 0 ? (
                   <p className="text-slate-500">كل المنتجات بحالة جيدة.</p>
                ) : (
                  <div className="overflow-x-auto">
                    <table className="w-full text-right">
                      <thead>
                        <tr className="text-slate-500 text-sm border-b">
                          <th className="pb-2">المنتج</th>
                          <th className="pb-2">المخزون الحالي</th>
                          <th className="pb-2">الحد الأدنى</th>
                        </tr>
                      </thead>
                      <tbody className="text-slate-700">
                        {products.filter(p => p.stock <= p.minStock).map(p => (
                          <tr key={p.id} className="border-b last:border-0">
                            <td className="py-2 font-medium">{p.name}</td>
                            <td className="py-2 text-red-600 font-bold">{p.stock}</td>
                            <td className="py-2">{p.minStock}</td>
                          </tr>
                        ))}
                      </tbody>
                    </table>
                  </div>
                )}
              </div>
            </div>
          </div>
        )}

        {/* --- REPORTS --- */}
        {currentTab === 'reports' && (
            <div className="space-y-6">
                <div className="flex justify-between items-center">
                    <h2 className="text-3xl font-bold text-slate-800">تقارير المبيعات</h2>
                    <div className="flex gap-2 items-center bg-white p-2 rounded-lg shadow-sm border">
                        <Calendar size={18} className="text-slate-500" />
                        <input 
                            type="date" 
                            value={reportStartDate} 
                            onChange={(e) => setReportStartDate(e.target.value)}
                            className="text-sm outline-none bg-transparent"
                        />
                        <span className="text-slate-400">-</span>
                        <input 
                            type="date" 
                            value={reportEndDate} 
                            onChange={(e) => setReportEndDate(e.target.value)}
                            className="text-sm outline-none bg-transparent"
                        />
                    </div>
                </div>

                {/* Summary Cards */}
                <div className="grid grid-cols-1 md:grid-cols-3 gap-6">
                    <div className="bg-white p-6 rounded-xl shadow-sm border border-slate-100">
                        <p className="text-slate-500 text-sm mb-1">مبيعات الفترة</p>
                        <h3 className="text-2xl font-bold text-emerald-600">{formatCurrency(reportData.totalSales)}</h3>
                    </div>
                    <div className="bg-white p-6 rounded-xl shadow-sm border border-slate-100">
                        <p className="text-slate-500 text-sm mb-1">عدد الفواتير</p>
                        <h3 className="text-2xl font-bold text-indigo-600">{reportData.totalInvoices}</h3>
                    </div>
                    <div className="bg-white p-6 rounded-xl shadow-sm border border-slate-100">
                        <p className="text-slate-500 text-sm mb-1">متوسط قيمة الفاتورة</p>
                        <h3 className="text-2xl font-bold text-blue-600">{formatCurrency(reportData.averageOrderValue)}</h3>
                    </div>
                </div>

                {/* Chart Section */}
                <div className="bg-white p-6 rounded-xl shadow-sm border border-slate-100">
                    <h3 className="text-lg font-bold mb-6 flex items-center gap-2">
                        <BarChart3 className="text-indigo-500 w-5 h-5" />
                        تحليل المبيعات اليومي
                    </h3>
                    <div className="bg-slate-50 rounded-lg p-4 border border-slate-100">
                        {renderChart()}
                    </div>
                </div>

                {/* Detailed List */}
                <div className="bg-white rounded-xl shadow-sm border border-slate-100 overflow-hidden">
                    <div className="p-4 border-b bg-slate-50">
                        <h3 className="font-bold text-slate-700">تفاصيل فواتير الفترة</h3>
                    </div>
                    <table className="w-full text-right">
                        <thead className="bg-white text-slate-500 text-xs font-semibold border-b">
                            <tr>
                                <th className="p-3">التاريخ</th>
                                <th className="p-3"># الفاتورة</th>
                                <th className="p-3">العميل</th>
                                <th className="p-3">القيمة</th>
                            </tr>
                        </thead>
                        <tbody className="divide-y divide-slate-100 text-sm">
                            {invoices
                              .filter(inv => {
                                 const start = new Date(reportStartDate); start.setHours(0,0,0,0);
                                 const end = new Date(reportEndDate); end.setHours(23,59,59,999);
                                 const d = inv.timestamp ? new Date(inv.timestamp) : new Date();
                                 return d >= start && d <= end;
                              })
                              .length === 0 ? (
                                <tr><td colSpan={4} className="p-4 text-center text-slate-400">لا توجد بيانات</td></tr>
                              ) : (
                                invoices
                                  .filter(inv => {
                                     const start = new Date(reportStartDate); start.setHours(0,0,0,0);
                                     const end = new Date(reportEndDate); end.setHours(23,59,59,999);
                                     const d = inv.timestamp ? new Date(inv.timestamp) : new Date();
                                     return d >= start && d <= end;
                                  })
                                  .map(inv => (
                                    <tr key={inv.id} className="hover:bg-slate-50">
                                        <td className="p-3 text-slate-500">{inv.date}</td>
                                        <td className="p-3 font-mono">#{inv.id.slice(-4)}</td>
                                        <td className="p-3">{inv.customerName}</td>
                                        <td className="p-3 font-bold text-emerald-600">{formatCurrency(inv.finalAmount)}</td>
                                    </tr>
                                  ))
                              )}
                        </tbody>
                    </table>
                </div>
            </div>
        )}

        {/* --- INVENTORY --- */}
        {currentTab === 'inventory' && (
          <div className="space-y-6">
             <div className="flex justify-between items-center">
               <h2 className="text-3xl font-bold text-slate-800">إدارة المخزون</h2>
               <button 
                 onClick={() => { setEditingProduct(null); setTempProduct({}); setShowProductModal(true); }}
                 className="bg-emerald-600 hover:bg-emerald-700 text-white px-4 py-2 rounded-lg flex items-center gap-2 shadow-sm"
               >
                 <Plus size={20} />
                 إضافة منتج
               </button>
             </div>

             <div className="bg-white p-4 rounded-xl shadow-sm border border-slate-100 flex gap-4">
               <div className="relative flex-1">
                 <Search className="absolute right-3 top-3 text-slate-400 w-5 h-5" />
                 <input 
                    type="text" 
                    placeholder="بحث عن منتج (الاسم أو الباركود)..." 
                    className="w-full pl-4 pr-10 py-2 border rounded-lg focus:outline-none focus:ring-2 focus:ring-emerald-500"
                    value={searchTerm}
                    onChange={(e) => setSearchTerm(e.target.value)}
                 />
               </div>
             </div>

             <div className="bg-white rounded-xl shadow-sm border border-slate-100 overflow-hidden">
                <table className="w-full text-right">
                  <thead className="bg-slate-50 text-slate-600 font-semibold border-b">
                    <tr>
                      <th className="p-4">اسم المنتج</th>
                      <th className="p-4">الباركود</th>
                      <th className="p-4">التصنيف</th>
                      <th className="p-4">المورد</th>
                      <th className="p-4">سعر الشراء</th>
                      <th className="p-4">سعر البيع</th>
                      <th className="p-4">المخزون</th>
                      <th className="p-4 text-center">إجراءات</th>
                    </tr>
                  </thead>
                  <tbody className="divide-y divide-slate-100">
                    {products.filter(p => p.name.includes(searchTerm) || (p.barcode && p.barcode.includes(searchTerm))).map(product => (
                      <tr key={product.id} className="hover:bg-slate-50 transition-colors">
                        <td className="p-4 font-medium text-slate-800">{product.name}</td>
                        <td className="p-4 font-mono text-xs text-slate-500">{product.barcode || '-'}</td>
                        <td className="p-4 text-slate-500 text-sm">
                          <span className="bg-slate-100 px-2 py-1 rounded-md">{product.category}</span>
                        </td>
                        <td className="p-4 text-slate-500 text-sm">
                           {getSupplierName(product.supplierId)}
                        </td>
                        <td className="p-4 text-slate-600">{formatCurrency(product.cost)}</td>
                        <td className="p-4 text-emerald-600 font-bold">{formatCurrency(product.price)}</td>
                        <td className="p-4">
                           <span className={`px-2 py-1 rounded-full text-sm font-bold ${product.stock <= product.minStock ? 'bg-red-100 text-red-600' : 'bg-green-100 text-green-600'}`}>
                             {product.stock}
                           </span>
                        </td>
                        <td className="p-4 flex justify-center gap-2">
                          <button 
                            onClick={() => { setEditingProduct(product); setTempProduct(product); setShowProductModal(true); }}
                            className="p-2 hover:bg-blue-50 text-blue-600 rounded-lg"
                          >
                            <Edit size={18} />
                          </button>
                          <button 
                            onClick={() => handleDeleteProduct(product.id)}
                            className="p-2 hover:bg-red-50 text-red-600 rounded-lg"
                          >
                            <Trash2 size={18} />
                          </button>
                        </td>
                      </tr>
                    ))}
                  </tbody>
                </table>
             </div>
          </div>
        )}

        {/* --- SALES (POS) --- */}
        {currentTab === 'sales' && (
          <div className="flex h-[calc(100vh-6rem)] gap-6">
             {/* Product Grid */}
             <div className="flex-1 flex flex-col gap-6">
                <div className="flex gap-4">
                  <div className="flex-1 bg-white p-4 rounded-xl shadow-sm border border-slate-100 relative">
                    <Search className="absolute right-6 top-6 text-slate-400 w-5 h-5" />
                    <input 
                        type="text" 
                        placeholder="بحث سريع عن منتج (الاسم أو الباركود)..." 
                        className="w-full px-4 pr-10 py-2 border rounded-lg focus:outline-none focus:ring-2 focus:ring-emerald-500"
                        value={searchTerm}
                        onChange={(e) => setSearchTerm(e.target.value)}
                    />
                  </div>
                </div>
                <div className="flex-1 overflow-y-auto grid grid-cols-2 lg:grid-cols-3 gap-4 pb-4">
                  {products.filter(p => p.name.includes(searchTerm) || (p.barcode && p.barcode.includes(searchTerm))).map(product => (
                    <button 
                      key={product.id}
                      onClick={() => addToCart(product)}
                      disabled={product.stock === 0}
                      className={`bg-white p-4 rounded-xl border transition-all text-right flex flex-col justify-between
                        ${product.stock === 0 ? 'opacity-50 cursor-not-allowed border-slate-200' : 'hover:border-emerald-500 hover:shadow-md border-slate-100'}`}
                    >
                      <div>
                        <h4 className="font-bold text-slate-800 mb-1">{product.name}</h4>
                        <div className="flex justify-between items-center mb-2">
                            <p className="text-xs text-slate-500">{product.category}</p>
                            {product.barcode && <p className="text-[10px] bg-slate-100 px-1 rounded text-slate-500 font-mono">{product.barcode}</p>}
                        </div>
                      </div>
                      <div className="flex justify-between items-end w-full">
                        <span className="text-emerald-600 font-bold">{formatCurrency(product.price)}</span>
                        <span className={`text-xs px-2 py-1 rounded-full ${product.stock > 0 ? 'bg-slate-100' : 'bg-red-100 text-red-600'}`}>
                          {product.stock > 0 ? `باقي: ${product.stock}` : 'نفد'}
                        </span>
                      </div>
                    </button>
                  ))}
                </div>
             </div>

             {/* Cart Sidebar */}
             <div className="w-96 bg-white rounded-xl shadow-lg border border-slate-100 flex flex-col h-full">
                <div className="p-5 border-b border-slate-100 bg-slate-50 rounded-t-xl">
                  <h3 className="font-bold text-lg flex items-center gap-2">
                    <ShoppingCart className="text-emerald-600" />
                    المبيعات الحالية
                  </h3>
                </div>
                
                <div className="flex-1 overflow-y-auto p-4 space-y-3">
                  {cart.length === 0 ? (
                    <div className="h-full flex flex-col items-center justify-center text-slate-400">
                      <ShoppingCart size={48} className="mb-2 opacity-20" />
                      <p>السلة فارغة</p>
                    </div>
                  ) : (
                    cart.map(item => (
                      <div key={item.id} className="flex justify-between items-center bg-slate-50 p-3 rounded-lg border border-slate-100">
                         <div className="flex-1 ml-2">
                           <h4 className="font-medium text-sm text-slate-800">{item.name}</h4>
                           <p className="text-xs text-emerald-600 font-bold">{formatCurrency(item.price * item.quantity)}</p>
                         </div>
                         <div className="flex items-center gap-2">
                            <button onClick={() => updateCartQuantity(item.id, -1)} className="w-6 h-6 bg-white border rounded flex items-center justify-center hover:bg-slate-100">-</button>
                            <span className="w-4 text-center text-sm font-bold">{item.quantity}</span>
                            <button onClick={() => updateCartQuantity(item.id, 1)} className="w-6 h-6 bg-white border rounded flex items-center justify-center hover:bg-slate-100">+</button>
                         </div>
                         <button onClick={() => removeFromCart(item.id)} className="mr-3 text-red-500 hover:text-red-700">
                           <X size={16} />
                         </button>
                      </div>
                    ))
                  )}
                </div>

                <div className="p-5 border-t border-slate-100 bg-slate-50 rounded-b-xl space-y-4">
                  <div className="relative">
                      <label className="text-sm text-slate-500 mb-1 block">العميل</label>
                      <div 
                          className="w-full border rounded-lg bg-white flex items-center relative"
                      >
                          <Search className="text-slate-400 w-4 h-4 absolute right-3 pointer-events-none" />
                          <input 
                              type="text"
                              className="w-full pr-9 pl-8 py-2 rounded-lg text-sm outline-none"
                              placeholder="ابحث عن عميل..."
                              value={isCustomerDropdownOpen ? customerSearchQuery : selectedCustomer}
                              onChange={(e) => {
                                  setCustomerSearchQuery(e.target.value);
                                  setIsCustomerDropdownOpen(true);
                              }}
                              onClick={() => {
                                  if (!isCustomerDropdownOpen) {
                                      setCustomerSearchQuery('');
                                      setIsCustomerDropdownOpen(true);
                                  }
                              }}
                          />
                          <button 
                              onClick={() => setIsCustomerDropdownOpen(!isCustomerDropdownOpen)}
                              className="absolute left-2 p-1 text-slate-400 hover:text-slate-600"
                          >
                              <ChevronDown size={16} />
                          </button>
                      </div>

                      {/* Dropdown List */}
                      {isCustomerDropdownOpen && (
                          <>
                              <div className="fixed inset-0 z-10" onClick={() => setIsCustomerDropdownOpen(false)}></div>
                              <div className="absolute top-full right-0 left-0 bg-white border rounded-lg shadow-lg mt-1 max-h-48 overflow-y-auto z-20">
                                  {customers
                                      .filter(c => c.name.toLowerCase().includes(customerSearchQuery.toLowerCase()))
                                      .map(c => (
                                          <button
                                              key={c.id}
                                              className={`w-full text-right px-4 py-2 text-sm hover:bg-emerald-50 hover:text-emerald-700 transition-colors ${selectedCustomer === c.name ? 'bg-emerald-50 text-emerald-700 font-bold' : 'text-slate-700'}`}
                                              onClick={() => {
                                                  setSelectedCustomer(c.name);
                                                  setIsCustomerDropdownOpen(false);
                                                  setCustomerSearchQuery('');
                                              }}
                                          >
                                              {c.name}
                                          </button>
                                      ))
                                  }
                                  {customers.filter(c => c.name.toLowerCase().includes(customerSearchQuery.toLowerCase())).length === 0 && (
                                       <div className="px-4 py-2 text-sm text-slate-400 text-center">لا يوجد عميل بهذا الاسم</div>
                                  )}
                              </div>
                          </>
                      )}
                  </div>
                   <div className="flex gap-2 items-center">
                     <span className="text-sm text-slate-500">خصم:</span>
                     <input 
                        type="number" 
                        className="w-full px-3 py-2 border rounded-lg text-sm bg-white"
                        value={invoiceDiscount}
                        onChange={(e) => setInvoiceDiscount(Number(e.target.value))}
                     />
                  </div>
                  
                  <div className="space-y-2 pt-2 border-t border-slate-200">
                    <div className="flex justify-between text-sm">
                      <span className="text-slate-500">المجموع الفرعي</span>
                      <span>{formatCurrency(cart.reduce((a, b) => a + b.price * b.quantity, 0))}</span>
                    </div>
                    <div className="flex justify-between text-sm text-red-500">
                      <span>الخصم</span>
                      <span>- {formatCurrency(invoiceDiscount)}</span>
                    </div>
                    <div className="flex justify-between text-xl font-bold text-slate-900 pt-2">
                      <span>الإجمالي</span>
                      <span>{formatCurrency(Math.max(0, cart.reduce((a, b) => a + b.price * b.quantity, 0) - invoiceDiscount))}</span>
                    </div>
                  </div>

                  <button 
                    onClick={handleSaveInvoice}
                    disabled={cart.length === 0}
                    className={`w-full py-3 rounded-lg font-bold shadow-md flex justify-center items-center gap-2
                      ${cart.length === 0 ? 'bg-slate-300 cursor-not-allowed text-slate-500' : 'bg-emerald-600 text-white hover:bg-emerald-700'}`}
                  >
                    <Save size={18} />
                    حفظ وطباعة
                  </button>
                </div>
             </div>
          </div>
        )}

        {/* --- INVOICES LIST --- */}
        {currentTab === 'invoices' && (
           <div className="space-y-6">
             <h2 className="text-3xl font-bold text-slate-800">سجل الفواتير</h2>
             <div className="bg-white rounded-xl shadow-sm border border-slate-100 overflow-hidden">
                <table className="w-full text-right">
                  <thead className="bg-slate-50 text-slate-600 font-semibold border-b">
                    <tr>
                      <th className="p-4"># الفاتورة</th>
                      <th className="p-4">العميل</th>
                      <th className="p-4">التاريخ</th>
                      <th className="p-4">عدد الأصناف</th>
                      <th className="p-4">الإجمالي</th>
                      <th className="p-4">إجراءات</th>
                    </tr>
                  </thead>
                  <tbody className="divide-y divide-slate-100">
                    {invoices.length === 0 ? (
                      <tr>
                        <td colSpan={6} className="p-8 text-center text-slate-400">لا توجد فواتير مسجلة بعد</td>
                      </tr>
                    ) : (
                      invoices.map((inv) => (
                        <tr key={inv.id} className="hover:bg-slate-50">
                          <td className="p-4 font-mono text-slate-600">#{inv.id.slice(-6)}</td>
                          <td className="p-4 font-medium">{inv.customerName}</td>
                          <td className="p-4 text-slate-500 text-sm">{inv.date}</td>
                          <td className="p-4 text-slate-500">{inv.items.length}</td>
                          <td className="p-4 font-bold text-emerald-600">{formatCurrency(inv.finalAmount)}</td>
                          <td className="p-4">
                            <button 
                              onClick={() => handlePrintInvoice(inv)}
                              className="text-blue-600 hover:bg-blue-50 p-2 rounded" 
                              title="طباعة"
                            >
                              <Printer size={18} />
                            </button>
                          </td>
                        </tr>
                      ))
                    )}
                  </tbody>
                </table>
             </div>
           </div>
        )}

        {/* --- CUSTOMERS TAB --- */}
        {currentTab === 'customers' && (
          <div className="space-y-6">
            <h2 className="text-3xl font-bold text-slate-800">إدارة العملاء</h2>
            
            {/* Add Customer Form */}
            <div className="bg-white p-4 rounded-xl shadow-sm border border-slate-100 flex gap-4 items-end">
              <div className="flex-1">
                <label className="text-sm text-slate-500">اسم العميل</label>
                <input type="text" className="w-full border rounded p-2" value={newCustomer.name || ''} onChange={e => setNewCustomer({...newCustomer, name: e.target.value})} />
              </div>
              <div className="flex-1">
                <label className="text-sm text-slate-500">رقم الهاتف</label>
                <input type="text" className="w-full border rounded p-2" value={newCustomer.phone || ''} onChange={e => setNewCustomer({...newCustomer, phone: e.target.value})} />
              </div>
              <div className="flex-1">
                <label className="text-sm text-slate-500">البريد الإلكتروني</label>
                <input type="email" className="w-full border rounded p-2" value={newCustomer.email || ''} onChange={e => setNewCustomer({...newCustomer, email: e.target.value})} />
              </div>
              <button onClick={handleAddCustomer} className="bg-emerald-600 text-white px-4 py-2 rounded h-10 hover:bg-emerald-700">إضافة</button>
            </div>

            {/* Customers Table */}
            <div className="bg-white rounded-xl shadow-sm border border-slate-100 overflow-hidden">
               <table className="w-full text-right">
                 <thead className="bg-slate-50 text-slate-600 font-semibold border-b">
                   <tr>
                     <th className="p-4">اسم العميل</th>
                     <th className="p-4">الهاتف</th>
                     <th className="p-4">البريد الإلكتروني</th>
                     <th className="p-4">إجراءات</th>
                   </tr>
                 </thead>
                 <tbody className="divide-y divide-slate-100">
                   {customers.map(c => (
                     <tr key={c.id}>
                       <td className="p-4">{c.name}</td>
                       <td className="p-4 font-mono text-slate-600">{c.phone || '-'}</td>
                       <td className="p-4 text-slate-500">{c.email || '-'}</td>
                       <td className="p-4">
                         {c.id !== '1' && (
                            <button onClick={() => deleteCustomer(c.id)} className="text-red-500 hover:bg-red-50 p-2 rounded">
                               <Trash2 size={18} />
                            </button>
                         )}
                       </td>
                     </tr>
                   ))}
                 </tbody>
               </table>
            </div>
          </div>
        )}

        {/* --- CATEGORIES TAB --- */}
        {currentTab === 'categories' && (
          <div className="space-y-6">
            <h2 className="text-3xl font-bold text-slate-800">التصنيفات</h2>
            
            <div className="bg-white p-4 rounded-xl shadow-sm border border-slate-100 flex gap-4 items-end">
              <div className="flex-1">
                <label className="text-sm text-slate-500">اسم التصنيف</label>
                <input type="text" className="w-full border rounded p-2" value={newCategory.name || ''} onChange={e => setNewCategory({...newCategory, name: e.target.value})} />
              </div>
              <div className="flex-2 w-1/2">
                <label className="text-sm text-slate-500">الوصف</label>
                <input type="text" className="w-full border rounded p-2" value={newCategory.description || ''} onChange={e => setNewCategory({...newCategory, description: e.target.value})} />
              </div>
              <button onClick={handleAddCategory} className="bg-emerald-600 text-white px-4 py-2 rounded h-10 hover:bg-emerald-700">إضافة</button>
            </div>

            <div className="bg-white rounded-xl shadow-sm border border-slate-100 overflow-hidden">
               <table className="w-full text-right">
                 <thead className="bg-slate-50 text-slate-600 font-semibold border-b">
                   <tr>
                     <th className="p-4">التصنيف</th>
                     <th className="p-4">الوصف</th>
                     <th className="p-4">إجراءات</th>
                   </tr>
                 </thead>
                 <tbody className="divide-y divide-slate-100">
                   {categories.map(c => (
                     <tr key={c.id}>
                       <td className="p-4 font-bold text-slate-700">{c.name}</td>
                       <td className="p-4 text-slate-500">{c.description}</td>
                       <td className="p-4">
                          <button onClick={() => deleteCategory(c.id)} className="text-red-500 hover:bg-red-50 p-2 rounded">
                             <Trash2 size={18} />
                          </button>
                       </td>
                     </tr>
                   ))}
                 </tbody>
               </table>
            </div>
          </div>
        )}

        {/* --- SUPPLIERS TAB --- */}
        {currentTab === 'suppliers' && (
          <div className="space-y-6">
            <h2 className="text-3xl font-bold text-slate-800">الموردين</h2>
            
            <div className="bg-white p-4 rounded-xl shadow-sm border border-slate-100 flex gap-4 items-end">
              <div className="flex-1">
                <label className="text-sm text-slate-500">اسم المورد / الشركة</label>
                <input type="text" className="w-full border rounded p-2" value={newSupplier.name || ''} onChange={e => setNewSupplier({...newSupplier, name: e.target.value})} />
              </div>
              <div className="flex-1">
                <label className="text-sm text-slate-500">الشخص المسؤول</label>
                <input type="text" className="w-full border rounded p-2" value={newSupplier.contactPerson || ''} onChange={e => setNewSupplier({...newSupplier, contactPerson: e.target.value})} />
              </div>
              <div className="flex-1">
                <label className="text-sm text-slate-500">رقم التواصل</label>
                <input type="text" className="w-full border rounded p-2" value={newSupplier.phone || ''} onChange={e => setNewSupplier({...newSupplier, phone: e.target.value})} />
              </div>
              <button onClick={handleAddSupplier} className="bg-emerald-600 text-white px-4 py-2 rounded h-10 hover:bg-emerald-700">إضافة</button>
            </div>

            <div className="bg-white rounded-xl shadow-sm border border-slate-100 overflow-hidden">
               <table className="w-full text-right">
                 <thead className="bg-slate-50 text-slate-600 font-semibold border-b">
                   <tr>
                     <th className="p-4">اسم الشركة</th>
                     <th className="p-4">المسؤول</th>
                     <th className="p-4">الهاتف</th>
                     <th className="p-4">إجراءات</th>
                   </tr>
                 </thead>
                 <tbody className="divide-y divide-slate-100">
                   {suppliers.map(s => (
                     <tr key={s.id}>
                       <td className="p-4 font-bold">{s.name}</td>
                       <td className="p-4 text-slate-600">{s.contactPerson}</td>
                       <td className="p-4 font-mono text-slate-500">{s.phone}</td>
                       <td className="p-4">
                          <button onClick={() => deleteSupplier(s.id)} className="text-red-500 hover:bg-red-50 p-2 rounded">
                             <Trash2 size={18} />
                          </button>
                       </td>
                     </tr>
                   ))}
                 </tbody>
               </table>
            </div>
          </div>
        )}

        {/* --- USERS TAB --- */}
        {currentTab === 'users' && (
          <div className="space-y-6">
            <h2 className="text-3xl font-bold text-slate-800">المستخدمين</h2>
            
            {/* Added: Full Name and Password fields */}
            <div className="bg-white p-4 rounded-xl shadow-sm border border-slate-100 grid grid-cols-1 md:grid-cols-2 lg:grid-cols-5 gap-4 items-end">
              <div>
                <label className="text-sm text-slate-500">الاسم الكامل</label>
                <input type="text" className="w-full border rounded p-2" value={newUser.fullName || ''} onChange={e => setNewUser({...newUser, fullName: e.target.value})} />
              </div>
              <div>
                <label className="text-sm text-slate-500">اسم المستخدم</label>
                <input type="text" className="w-full border rounded p-2" value={newUser.username || ''} onChange={e => setNewUser({...newUser, username: e.target.value})} />
              </div>
              <div>
                <label className="text-sm text-slate-500">كلمة المرور</label>
                <input type="password" className="w-full border rounded p-2" value={newUser.password || ''} onChange={e => setNewUser({...newUser, password: e.target.value})} />
              </div>
              <div>
                <label className="text-sm text-slate-500">الصلاحية</label>
                <select className="w-full border rounded p-2 bg-white" value={newUser.role} onChange={e => setNewUser({...newUser, role: e.target.value as 'admin' | 'staff'})}>
                   <option value="staff">موظف (Staff)</option>
                   <option value="admin">مدير (Admin)</option>
                </select>
              </div>
              <button onClick={handleAddUser} className="bg-emerald-600 text-white px-4 py-2 rounded h-10 hover:bg-emerald-700">إضافة</button>
            </div>

            <div className="bg-white rounded-xl shadow-sm border border-slate-100 overflow-hidden">
               <table className="w-full text-right">
                 <thead className="bg-slate-50 text-slate-600 font-semibold border-b">
                   <tr>
                     <th className="p-4">الاسم الكامل</th>
                     <th className="p-4">اسم المستخدم</th>
                     <th className="p-4">الصلاحية</th>
                     <th className="p-4">إجراءات</th>
                   </tr>
                 </thead>
                 <tbody className="divide-y divide-slate-100">
                   {users.map(u => (
                     <tr key={u.id}>
                       <td className="p-4 font-medium">{u.fullName}</td>
                       <td className="p-4 text-slate-600">{u.username}</td>
                       <td className="p-4">
                          <span className={`px-2 py-1 rounded text-xs ${u.role === 'admin' ? 'bg-purple-100 text-purple-700' : 'bg-blue-100 text-blue-700'}`}>
                            {u.role === 'admin' ? 'مدير' : 'موظف'}
                          </span>
                       </td>
                       <td className="p-4">
                          {u.id !== '1' && (
                            <button onClick={() => deleteUser(u.id)} className="text-red-500 hover:bg-red-50 p-2 rounded">
                               <Trash2 size={18} />
                            </button>
                          )}
                       </td>
                     </tr>
                   ))}
                 </tbody>
               </table>
            </div>
          </div>
        )}

        {/* --- SETTINGS TAB --- */}
        {currentTab === 'settings' && (
          <div className="space-y-6">
            <h2 className="text-3xl font-bold text-slate-800">إعدادات المتجر</h2>
            
            <div className="bg-white p-6 rounded-xl shadow-sm border border-slate-100 max-w-3xl">
               <div className="space-y-6">
                  {/* Store Name */}
                  <div className="flex gap-4 items-start border-b pb-6 border-slate-100">
                     <div className="p-3 bg-blue-50 rounded-lg text-blue-600 mt-1">
                        <Store size={24} />
                     </div>
                     <div className="flex-1">
                        <label className="block font-bold text-slate-700 mb-2">اسم المتجر</label>
                        <p className="text-sm text-slate-500 mb-3">سيظهر هذا الاسم في أعلى الفواتير والتقارير.</p>
                        <input 
                           type="text" 
                           className="w-full border rounded-lg p-3 bg-slate-50 focus:bg-white transition-colors"
                           value={storeSettings.name}
                           onChange={(e) => setStoreSettings({...storeSettings, name: e.target.value})}
                        />
                     </div>
                  </div>

                  {/* Address */}
                  <div className="flex gap-4 items-start border-b pb-6 border-slate-100">
                     <div className="p-3 bg-emerald-50 rounded-lg text-emerald-600 mt-1">
                        <MapPin size={24} />
                     </div>
                     <div className="flex-1">
                        <label className="block font-bold text-slate-700 mb-2">العنوان</label>
                        <p className="text-sm text-slate-500 mb-3">عنوان المتجر الذي سيظهر في الفواتير.</p>
                        <textarea 
                           className="w-full border rounded-lg p-3 bg-slate-50 focus:bg-white transition-colors h-24 resize-none"
                           value={storeSettings.address}
                           onChange={(e) => setStoreSettings({...storeSettings, address: e.target.value})}
                        />
                     </div>
                  </div>

                  {/* Currency */}
                  <div className="flex gap-4 items-start border-b pb-6 border-slate-100">
                     <div className="p-3 bg-amber-50 rounded-lg text-amber-600 mt-1">
                        <Coins size={24} />
                     </div>
                     <div className="flex-1">
                        <label className="block font-bold text-slate-700 mb-2">العملة</label>
                        <p className="text-sm text-slate-500 mb-3">رمز العملة المستخدم في النظام (مثال: MAD, USD, SAR, AED).</p>
                        <div className="flex gap-2">
                            <select 
                              className="w-1/3 border rounded-lg p-3 bg-slate-50 focus:bg-white"
                              value={storeSettings.currency}
                              onChange={(e) => setStoreSettings({...storeSettings, currency: e.target.value})}
                            >
                                <option value="MAD">MAD - درهم مغربي</option>
                                <option value="SAR">SAR - ريال سعودي</option>
                                <option value="AED">AED - درهم إماراتي</option>
                                <option value="EGP">EGP - جنيه مصري</option>
                                <option value="USD">USD - دولار أمريكي</option>
                                <option value="EUR">EUR - يورو</option>
                            </select>
                            <input 
                                type="text" 
                                placeholder="أو اكتب رمزاً خاصاً"
                                className="flex-1 border rounded-lg p-3 bg-slate-50 focus:bg-white"
                                value={storeSettings.currency}
                                onChange={(e) => setStoreSettings({...storeSettings, currency: e.target.value})}
                            />
                        </div>
                     </div>
                  </div>

                  {/* Phones */}
                  <div className="flex gap-4 items-start">
                     <div className="p-3 bg-purple-50 rounded-lg text-purple-600 mt-1">
                        <Phone size={24} />
                     </div>
                     <div className="flex-1">
                        <label className="block font-bold text-slate-700 mb-2">أرقام التواصل</label>
                        <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                           <div>
                              <label className="text-xs text-slate-500 mb-1 block">رقم الهاتف 1 (أساسي)</label>
                              <input 
                                 type="text" 
                                 className="w-full border rounded-lg p-3 bg-slate-50 focus:bg-white"
                                 value={storeSettings.phone1}
                                 onChange={(e) => setStoreSettings({...storeSettings, phone1: e.target.value})}
                              />
                           </div>
                           <div>
                              <label className="text-xs text-slate-500 mb-1 block">رقم الهاتف 2 (اختياري)</label>
                              <input 
                                 type="text" 
                                 className="w-full border rounded-lg p-3 bg-slate-50 focus:bg-white"
                                 value={storeSettings.phone2}
                                 onChange={(e) => setStoreSettings({...storeSettings, phone2: e.target.value})}
                              />
                           </div>
                        </div>
                     </div>
                  </div>
               </div>
            </div>
          </div>
        )}
      </main>

      {/* --- ADD/EDIT PRODUCT MODAL --- */}
      {showProductModal && (
        <div className="fixed inset-0 bg-black/50 z-50 flex items-center justify-center backdrop-blur-sm">
          <div className="bg-white rounded-xl shadow-2xl w-full max-w-lg p-6 animate-scale-in">
            <h3 className="text-xl font-bold mb-4 text-slate-800">
              {editingProduct ? 'تعديل المنتج' : 'إضافة منتج جديد'}
            </h3>
            
            <div className="space-y-4">
              {/* Product Name & Barcode Row */}
              <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                <div className="md:col-span-2">
                   <label className="block text-sm font-medium text-slate-700 mb-1">الباركود (اختياري)</label>
                   <div className="relative">
                       <input 
                         type="text" 
                         className="w-full border rounded-lg p-2 pl-10 focus:ring-2 focus:ring-emerald-500 outline-none font-mono"
                         placeholder="امسح الباركود..."
                         value={tempProduct.barcode || ''} 
                         onChange={e => setTempProduct({...tempProduct, barcode: e.target.value})}
                       />
                       <ScanBarcode className="absolute left-3 top-2.5 text-slate-400 w-5 h-5" />
                   </div>
                </div>
                <div className="md:col-span-2">
                  <label className="block text-sm font-medium text-slate-700 mb-1">اسم المنتج</label>
                  <input 
                    type="text" 
                    className="w-full border rounded-lg p-2 focus:ring-2 focus:ring-emerald-500 outline-none"
                    value={tempProduct.name || ''} 
                    onChange={e => setTempProduct({...tempProduct, name: e.target.value})}
                  />
                </div>
              </div>
              
              <div className="grid grid-cols-2 gap-4">
                <div>
                  <label className="block text-sm font-medium text-slate-700 mb-1">التصنيف</label>
                  <select 
                    className="w-full border rounded-lg p-2 focus:ring-2 focus:ring-emerald-500 outline-none bg-white"
                    value={tempProduct.category || ''} 
                    onChange={e => setTempProduct({...tempProduct, category: e.target.value})}
                  >
                    <option value="">اختر التصنيف</option>
                    {categories.map(c => <option key={c.id} value={c.name}>{c.name}</option>)}
                  </select>
                </div>
                <div>
                  <label className="block text-sm font-medium text-slate-700 mb-1">المخزون الحالي</label>
                  <input 
                    type="number" 
                    className="w-full border rounded-lg p-2 focus:ring-2 focus:ring-emerald-500 outline-none"
                    value={tempProduct.stock || ''} 
                    onChange={e => setTempProduct({...tempProduct, stock: Number(e.target.value)})}
                  />
                </div>
              </div>

              <div className="grid grid-cols-2 gap-4">
                <div>
                  <label className="block text-sm font-medium text-slate-700 mb-1">سعر الشراء (التكلفة)</label>
                  <input 
                    type="number" 
                    className="w-full border rounded-lg p-2 focus:ring-2 focus:ring-emerald-500 outline-none"
                    value={tempProduct.cost || ''} 
                    onChange={e => setTempProduct({...tempProduct, cost: Number(e.target.value)})}
                  />
                </div>
                <div>
                  <label className="block text-sm font-medium text-slate-700 mb-1">سعر البيع</label>
                  <input 
                    type="number" 
                    className="w-full border rounded-lg p-2 focus:ring-2 focus:ring-emerald-500 outline-none"
                    value={tempProduct.price || ''} 
                    onChange={e => setTempProduct({...tempProduct, price: Number(e.target.value)})}
                  />
                </div>
              </div>
              
              <div>
                  <label className="block text-sm font-medium text-slate-700 mb-1">المورد (اختياري)</label>
                  <select 
                    className="w-full border rounded-lg p-2 focus:ring-2 focus:ring-emerald-500 outline-none bg-white"
                    value={tempProduct.supplierId || ''} 
                    onChange={e => setTempProduct({...tempProduct, supplierId: e.target.value})}
                  >
                    <option value="">اختر المورد</option>
                    {suppliers.map(s => <option key={s.id} value={s.id}>{s.name}</option>)}
                  </select>
              </div>
              
              <div>
                  <label className="block text-sm font-medium text-slate-700 mb-1">تنبيه عند وصول العدد إلى</label>
                  <input 
                    type="number" 
                    className="w-full border rounded-lg p-2 focus:ring-2 focus:ring-emerald-500 outline-none"
                    value={tempProduct.minStock || ''} 
                    onChange={e => setTempProduct({...tempProduct, minStock: Number(e.target.value)})}
                  />
              </div>
            </div>

            <div className="flex justify-end gap-3 mt-8">
              <button 
                onClick={() => setShowProductModal(false)}
                className="px-4 py-2 text-slate-500 hover:bg-slate-100 rounded-lg"
              >
                إلغاء
              </button>
              <button 
                onClick={handleSaveProduct}
                className="px-6 py-2 bg-emerald-600 text-white rounded-lg hover:bg-emerald-700 font-medium"
              >
                حفظ البيانات
              </button>
            </div>
          </div>
        </div>
      )}

      {/* --- DELETE CONFIRMATION MODAL --- */}
      {showDeleteModal && (
        <div className="fixed inset-0 bg-black/50 z-50 flex items-center justify-center backdrop-blur-sm">
          <div className="bg-white rounded-xl shadow-2xl w-full max-w-sm p-6 animate-scale-in">
            <div className="text-center">
              <div className="w-12 h-12 bg-red-100 rounded-full flex items-center justify-center mx-auto mb-4">
                <AlertTriangle className="text-red-600 w-6 h-6" />
              </div>
              <h3 className="text-xl font-bold text-slate-800 mb-2">تأكيد الحذف</h3>
              <p className="text-slate-500 mb-6">هل أنت متأكد من رغبتك في حذف هذا المنتج؟ لا يمكن التراجع عن هذا الإجراء.</p>
              
              <div className="flex gap-3 justify-center">
                <button 
                  onClick={() => setShowDeleteModal(false)}
                  className="px-4 py-2 text-slate-600 hover:bg-slate-100 rounded-lg font-medium"
                >
                  إلغاء
                </button>
                <button 
                  onClick={confirmDeleteProduct}
                  className="px-6 py-2 bg-red-600 text-white rounded-lg hover:bg-red-700 font-medium"
                >
                  نعم، احذف
                </button>
              </div>
            </div>
          </div>
        </div>
      )}
    </div>
  );
};

export default App;